classdef AdjacencyMatrix < handle
    %ADJACENCYMATRIX Class to store Adjacency Matrix of a biological neural
    %network
    %   adjMat = AdjacencyMatrix('file',filename,'delimiter',delimiter)
    %       will load the adjacency matrix from a file. The file should be
    %       a text file with columns delimited by a reserved character and
    %       rows delimited by newline characters.  The default column
    %       delimiter is ',' but this can be modified by specifying the
    %       'delimiter' parameter. The first line of the file should have
    %       an empty entry, followed by all of the column labels.  From the
    %       second line onwards, the first entry should be a row label,
    %       followed by the adjacency matrix values separated by a
    %       delimiter.
    %   adjMat = AdjacencyMatrix(matrix,rowLabels,columnLabels) will load
    %       the adjacency matrix from MATLAB array 'matrix'.  rowLabels and
    %       columnLabels should be cell arrays of strings giving the labels
    %       for rows and columns.
    %   adjMat = AdjacencyMatrix( ... ,'matrixType',matrixtype) will
    %       specify the type of matrix.  This is for reference purposes
    %       only, and does not affect any computations.  This will appear
    %       in the disp() function for AdjacencyMatrix
    %
    %   Note: the AdjacencyMatrix constructor will purge the matrix of
    %   empty rows and columns.
    
    properties (Access = protected)
        % cells is an array of structs with the properties of cell which
        % gives the cell names and their respective row and column numbers
        % in the matrix A
        % cells has subfields:
        % cells.name, cells.rowNumber, cells.columnNumber, cells.type,
        % cells.group
        cells = [];
        
        %colIndices and rowIndices gives the index in cells corresponding
        %to a row number and column number in A.  This is the inverse
        %function of cells.rowNumber and cells.columnNumber.  So
        %cells(columnIndices(x)).columnNumber = x
        columnIndices = [];
        rowIndices = [];
        
        %columnLabels and rowLabels are just the copies of these arguments
        %passed in the constructor
        columnLabels = {''};
        rowLabels = {''};
        % A is a sparse matrix storing the connectivity data. Its rows
        % and columns are indexed by cells.
        A = sparse([]);
        
        % type is an optional variable representing the type of this
        % matrix, for example 'chemical' or 'electrical'
        matrixType;
        
        %variables to store betweenness centrality information
        BC;
        BClabels;
    end
    
    properties (Constant)
        sensory_male = {'PHAL','PHAR','PHBL','PHBR','PHCL','PHCR','R1AL',...
            'R1AR','R1BL','R1BR','R2AL','R2AR','R2BL','R2BR','R3AL','R3AR',...
            'R3BL','R3BR','R4AL','R4AR','R4BL','R4BR','R5AL','R5AR','R5BL',...
            'R5BR','R6AL','R6AR','R6BL','R6BR','R7AL','R7AR','R7BL','R7BR',...
            'R8AL','R8AR','R8BL','R8BR','R9AL','R9AR','R9BL','R9BR','HOA',...
            'HOB','PCAL','PCAR','PCBL','PCBR','PCCL','PCCR','SPCL','SPCR',...
            'SPDL','SPDR','SPVL','SPVR','ALNL','ALNR','PLML','PLMR','PLNL',...
            'PLNR','PQR','PVM','PVDL','PVDR','PDEL','PDER','PVR','AVG'};
        inter_male = {'PVY','AVAL','AVAR','AVBL','AVBR','AVDL','AVDR',...
            'PVCL','PVCR','AVKL','AVKR','AVL','CP07','CP08','CP09','PDA',...
            'PDB','PDC','PVX','PVV','LUAL','LUAR','DX1','DX2','DX3','CA02',...
            'CA03','CA04','CA05','CA06','CA07','CA08','CA09','CP01','CP02','CP03',...
            'CP04','CP05','CP06','DVA','DVB','DVC','DVE','DVF','PGA','EF1',...
            'EF2','EF3','AN3a','AN3b','PVNL','PVNR','PVWL','PVWR','PVS',...
            'PVU','PVT','AN1a','AN1b','AN2a','AN2b','PVQL','PVQR'};
        motor_male = {'PVZ','DA04','DA05','DA06','DA07','DA08','DA09',...
            'DB03','DB04','DB05','DB06','DB07','DD03','DD04','DD05','DD06',...
            'AS08','AS09','AS10','AS11','VA08','VA09','VA10','VA11','VA12',...
            'VB04','VB05','VB06','VB07','VB08','VB09','VB10','VB11','VD09',...
            'VD10','VD11','VD12','VD13'}
        muscles_male = {'dBWML21','dBWMR21','dBWML22','dBWMR22','dBWML23',...
            'dBWMR23','dBWML24','dBWMR24','vBWML17','vBWMR18','vBWML19',...
            'vBWMR19','vBWML20','vBWMR20','vBWML21','vBWMR21','vBWML22',...
            'vBWMR22','vBWML23','vBWMR23','vBWMR24','cdlL','cdlR','polL',...
            'polR','dglR8','dglL7','dglR7','dglL6','dglR6','dglL5',...
            'dglR5','dglL4','dglR4','dglL3','dglR3','dglL2','dglR2',...
            'dglL1','dglR1','gecL','gecR','grtL','grtR','aobL','aobR',...
            'pobL','pobR','adp','dspL','dspR','dsrL','dsrR','vspL','vspR',...
            'vsrL','vsrR','ailL','ailR','pilL','pilR','intL','intR','sph',...
            'gonad','hyp'};
        
        % Note: hermaphrodite classifications are not mutually exclusive!!
        sensory_herm = {'IL2DL', 'IL2VL', 'IL2L', 'URADL', 'IL1VL', 'IL2DR', 'IL1DL', 'OLLL', 'IL1L', 'URYDL', 'OLQDL', 'URYDR', 'IL2R', 'URYVL', 'OLLR', 'URBL', 'IL2VR', 'IL1DR', 'URADR', 'IL1R', 'URAVL', 'OLQVL', 'URBR', 'OLQDR', 'URYVR', 'BAGL', 'CEPVL', 'BAGR', 'OLQVR', 'URAVR', 'IL1VR', 'CEPVR', 'CEPDL', 'URXL', 'CEPDR', 'ASKL', 'URXR', 'AFDL', 'ASKR', 'ADLL', 'ADFL', 'AFDR', 'AWBL', 'AWCL', 'ADFR', 'ASGL', 'ADLR', 'AWAL', 'AWBR', 'ASIL', 'ASHL', 'ASGR', 'ASHR', 'AWCR', 'AWAR', 'ASIR', 'ASEL', 'AUAL', 'ASER', 'ASJL', 'AUAR', 'ASJR', 'FLPL', 'FLPR', 'AQR', 'ADEL', 'ADER', 'AVG', 'AVM', 'ALML', 'ALMR', 'PDEL', 'PDER', 'PVM', 'PHAL', 'PHAR', 'PLNR', 'PHBR', 'ALNL', 'PHBL', 'ALNR', 'PQR', 'PVR', 'PLNL', 'PHCR', 'PHCL', 'PLMR', 'PLML'};
        inter_herm = {'RIPL', 'RIPR', 'SAAVL', 'URXL', 'ALA', 'AVAL', 'RIAL', 'SAAVR', 'URXR', 'AVAR', 'RIAR', 'AVEL', 'SIBDL', 'RIH', 'AVER', 'SAADL', 'SIBDR', 'AIBL', 'SIBVL', 'RIVL', 'SAADR', 'AVHL', 'AVHR', 'RIVR', 'AIBR', 'RIBL', 'AVBL', 'SIBVR', 'AVJR', 'SIADL', 'AVJL', 'AVBR', 'RIBR', 'AIAL', 'RIR', 'AVDR', 'AVDL', 'AINL', 'AINR', 'SIADR', 'AVL', 'RICL', 'AIAR', 'AIZR', 'SIAVL', 'SIAVR', 'RICR', 'AIZL', 'AIYR', 'AIMR', 'AIML', 'RIS', 'AIYL', 'AVKR', 'AVKL', 'AVFR', 'SABVL', 'SABVR', 'AVFL', 'RIFR', 'ADAR', 'ADAL', 'RIGR', 'RIFL', 'SABD', 'RIGL', 'BDUR', 'BDUL', 'SDQR', 'SDQL', 'PVDL', 'PVDR', 'PVPR', 'PVT', 'PVPL', 'DVA', 'DVC', 'PVQR', 'PVQL', 'LUAL', 'PVCL', 'LUAR', 'PVCR', 'PVWL', 'PVWR', 'PVNR', 'PVNL'};
        motor_herm = {'URADL', 'IL1VL', 'IL1DL', 'IL1L', 'IL1DR', 'URADR', 'IL1R', 'URAVL', 'RMED', 'RMEL', 'URAVR', 'RMER', 'IL1VR', 'RID', 'RMEV', 'RMDVL', 'SMDVL', 'RMDVR', 'RMDL', 'SMDVR', 'RMDR', 'RMDDL', 'RIVL', 'SMDDL', 'RMHL', 'RMDDR', 'RIVR', 'RMFL', 'RMHR', 'RMFR', 'SMDDR', 'SMBDL', 'RIML', 'RIMR', 'SMBVR', 'SMBVL', 'AVL', 'SMBDR', 'VB02', 'VB01', 'DB02', 'RMGL', 'RMGR', 'VA01', 'DD01', 'DB01', 'AS01', 'VD01', 'DA01', 'VD02', 'VA02', 'VB03', 'AS02', 'DB03', 'DA02', 'VD03', 'VA03', 'VB04', 'VC01', 'DD02', 'AS03', 'VD04', 'DA03', 'VA04', 'VB05', 'VC02', 'DB04', 'AS04', 'VD05', 'VA05', 'VB06', 'DA04', 'DD03', 'VC03', 'AS05', 'VD06', 'VA06', 'VB07', 'DB05', 'AS06', 'VD07', 'VC04', 'DA05', 'VC05', 'HSNR', 'HSNL', 'VA07', 'VB08', 'AS07', 'DD04', 'VD08', 'VA08', 'VB09', 'DB06', 'AS08', 'VD09', 'DA06', 'VA09', 'VB10', 'AS09', 'DD05', 'VD10', 'VA10', 'VB11', 'DB07', 'AS10', 'DA07', 'VD11', 'VA11', 'AS11', 'VD12', 'VA12', 'DD06', 'DA08', 'DA09', 'PDB', 'VD13', 'PDA', 'DVB'};
        
        
        class5 = {'AVNL','AVNR','PLML','PLMR','PLNL','PLNR','PQR','PVM','PVDL',...
            'PVDR','PDEL','PDER','PVR','PHCL','PHCR','AVKL','AVKR','DVA',...
            'PVWL','PVWR','AN1a','AN1b','AN2a','AN2b','PVQL','PVQR','ALNL','ALNR'};
        class4 = {'AN3a','AN3b','PVNL','PVNR'};
        class2 = {'EF1','EF2','EF3','AVG','AVL','PVS','PVT','PVU'};
        class1 = {'AS11','AVAL','AVAR','AVBL','AVBR','AVDL','AVDR','CA03','CA04','CA05','CA06','CA07','CA08','CA09','CP01','CP02','CP03','CP04','CP05','CP06','CP07','CP08','CP09','DA07','DA08','DA09','DB05','DD03','DD06','DVB','DVC','DVE','DVF','DX1','DX2','DX3','HOA','HOB','LUAL','LUAR','PCAL','PCAR','PCBL','PCBR','PCCL','PCCR','PDA','PDB','PDC','PGA','PHAL','PHAR','PHBL','PHBR','PVCL','PVCR','PVV','PVX','PVY','PVZ','R1AL','R1AR','R1BL','R1BR','R2AL','R2AR','R2BL','R2BR','R3AL','R3AR','R3BL','R3BR','R4AL','R4AR','R4BL','R4BR','R5AL','R5AR','R5BL','R5BR','R6AL','R6AR','R6BL','R6BR','R7AL','R7AR','R7BL','R7BR','R8AL','R8AR','R8BL','R8BR','R9AL','R9AR','R9BL','R9BR','SPCL','SPCR','SPDL','SPDR','SPVL','SPVR','VA09','VA10','VA11','VA12','VB05','VB06','VB07','VB08','VB09','VB10','VB11','VD09','VD10','VD11','VD12','VD13','AS08','AS09','AS10','CA02','DA04','DA05','DA06','DB04','DB06','DB07','DD04','DD05','adp','ailL','ailR','aobL','aobR','cdlL','cdlR','dBWML21','dBWML22','dBWML23','dBWML24','dBWMR21','dBWMR22','dBWMR23','dBWMR24','dglL1','dglL2','dglL4','dglL5','dglL6','dglL7','dglR1','dglR2','dglR3','dglR5','dglR6','dglR7','dglR8','dspL','dspR','gecL','gecR','gonad','grtL','grtR','hyp','intL','intR','pobL','pobR','polL','polR','sph','vBWML17','vBWML19','vBWML20','vBWML21','vBWML22','vBWML23','vBWMR18','vBWMR19','vBWMR20','vBWMR21','vBWMR22','vBWMR23','vBWMR24','vspL','vspR'};
    end
    
    methods (Static = true)
        [A rowLabels columnLabels] = fillFromFile(fileName,delimiter)
        writeToFile(fileName,A,rowLabels,columnLabels)
        writeToPajek(CIJ,labels,fname,arcs)
        
        muscleGroups = getMuscleGroups()
        tripletcounts = countTriplets(A)
        doubletCounts = countDoublets(A)
        
    end
    
    methods (Access = public)
        
        
        function obj = AdjacencyMatrix(varargin)
            
            % load says whether we are ready to process the matrix A
            load = false;
            
            %set default value for matrixType
            obj.matrixType = 'unknown';
            
            % Check if we are loading from file; if so, the second argument
            % should be the filename, and the following arguments set
            % paremeters
            if strcmp(varargin{1},'file')
                %Set default value for delimiter
                delimiter = ',';
                for i = 3:2:nargin
                    switch varargin{i}
                        case 'delimiter'
                            delimiter = varargin{i+1};
                        case 'matrixType'
                            obj.matrixType = varargin{i+1};
                    end  
                end              
                [A rowLabels columnLabels] = AdjacencyMatrix.fillFromFile(varargin{2},delimiter);
                load = true;
            end
            
            
            % If the first argument is a numeric matrix, then we are
            % loading directly from a matrix. In this case the second
            % argument should be the rowLabels, and the third argument
            % should be the columnLabels, and the fourth and fifth
            % arguments should be the matrix type parameter / value
            if(nargin >= 3 && isnumeric(varargin{1}))
                A = varargin{1};
                
                
                % Check to make sure rowLabels and columnLabels are the
                % correct dimensionality
                rowLabels = varargin{2};
                columnLabels = varargin{3};
                rowLabels = reshape(rowLabels,length(rowLabels),1);
                columnLabels = reshape(columnLabels,1,length(columnLabels));
                
                
                %                 if size(rowLabels,2) ~= 1
                %                     if size(rowLabels,1) == 1
                %                         rowLabels = rowLabels';
                %                     else
                %                         error('AdjacencyMatrix:ConstructorError','Parameter rowLabels must be 1xN or Nx1 cell array of strings');
                %                     end
                %                 end
                %
                
                %                 if size(columnLabels,1) ~= 1
                %                     if size(columnLabels,2) == 1
                %                         columnLabels = columnLabels';
                %                     else
                %                         error('AdjacencyMatrix:ConstructorError','Parameter columnLabels must be Nx1 or 1xN cell array of strings');
                %                     end
                %                 end
                
                % Loop through the rest of the arguments, setting
                % parameters if necessary
                for i = 4:2:nargin
                    switch varargin{i}
                        case 'matrixType'
                            obj.matrixType = varargin{i+1};
                    end
                end
                
                load = true;
            end
            
            
            
            if load
                % Change NaN's to zero in A
                A(isnan(A)) = 0;
                % Purge A of empty rows
                rowLabels = rowLabels(sum(A,2)~=0);
                % Purge A of empty columns
                columnLabels = columnLabels(sum(A)~=0);
                
                if (length(rowLabels) == 0) || (length(columnLabels) == 0)
                    error('AdjacencyMatrix:ConstructorError','Empty adjacency matrix');
                end
                
                A = A(sum(A,2) ~= 0,sum(A) ~= 0);
                
                %                 fprintf('length of rowlabels %u\n',length(rowLabels));
                obj.cells = struct('name',rowLabels','rowNumber',num2cell(1:length(rowLabels)),'columnNumber',0,'type','cell','group','');
                
                obj.rowIndices = 1:length(rowLabels);
                
                % Now loop through columnLabels to fill out the columnNumber
                % fields of the struct array cells
                % If there are column names which did not appear in
                % rowNames then we need to create new members of cells to
                % contain these
                count = numel(obj.cells) + 1;
                
                for ii = 1:numel(columnLabels)
                    idx = strcmp({obj.cells.name},columnLabels{ii});
                    
                    if any(idx)
                        obj.cells(idx).columnNumber = ii;
                        obj.columnIndices(ii) = find(idx);
                    else
                        
                        % If this cell only exists on the postsynaptic side
                        obj.cells(count).name = columnLabels{ii};
                        obj.cells(count).rowNumber = 0;
                        obj.cells(count).columnNumber = ii;
                        obj.cells(count).type = 'cell';
                        
                        obj.columnIndices(ii) = count;
                        count = count + 1;
                    end
                    
                end
                obj.A = sparse(A);
                
                obj.rowLabels = rowLabels;
                obj.columnLabels = columnLabels;
                
            end
            
        end
        
        % Overload the default disp function to give information about the
        % adjacency matrix object
        function disp(obj)
            disp(['AdjacencyMatrix object of type ' obj.matrixType])
            fprintf('%i presynaptic cells\n',numel(obj.rowLabels));
            fprintf('%i postsynaptic cells\n',numel(obj.columnLabels));
            fprintf('%i entries in adjacency matrix\n',nnz(obj.A));
        end
        
        
        % Functions defined in external files
        
        %% Functions for manipulating and retrieving AdjacencyMatrix entries
        
        % Functions to overload matlab operators
        [B varargout] = subsref(A,S)
        sumMatrix = plus(mat1,mat2)
        prodMat = times(a,b)
        diffMat = minus(mat1, mat2)
        
        % Shows all connections in this adjacency matrix
        showConnections(obj)
        
        % get connected components - this will return a cell array of
        % AdjacencyMatrix objects
        components = getConnectedComponents(obj,varargin)
        
        % get a subset of this AdjacencyMatrix, specifying only particular
        % neurons
        [subMatrix newA] = getSubMatrix(obj,varargin)
        
        % get this AdjacencyMatrix object in square matrix form with
        % identical labels on row and column axes
        [A labels] = getSquareMatrix(obj)
        
        
        % Function to filter out weak connections
        newMat = threshold(obj,N,varargin)
        
        %Function to filter self-edges
        newMat = rmSelfEdges(adjMat)
        
        function [A rowLabels columnLabels] = getMatrix(obj)
            A = obj.A; rowLabels = obj.rowLabels; columnLabels = obj.columnLabels;
        end
        
        % Combine cells into group to make meta-nodes in the
        % AdjacencyMatrix object
        reducedMat = combineCells(obj,varargin)
       
        %% Graph Statistic functions
        C = clusteringCoefficient(varargin)
        C = charPathLength(obj,varargin)
        [b varargout] = betweennessCentrality(obj, varargin)
  
        %% Community structure functions
        [commLabels Q reducedMat] = getCommunitiesMM(obj,varargin)
        
        % These functions find sub-communities using a qCut style
        % statistical test for significance
        matrices = getCommunitiesHMM(obj)
        comms = getCommunitiesMMsig( obj,minQ,minZ,nIters)
        
        % Calculate communities using simulated annealing algorithm
        [commLabels Q reducedMat] = getCommunitiesMMSA(obj,varargin)
        
        % Calculate communities using link communities algorithm
        [commLabelsLinks commLabelsNodes] = getLinkCommunities(obj,varargin)
        
        % Show functional cartography, according to Guimera and Amaral,
        % Nature, Feb. 2005.
        [M labels] = functionalCartography(obj, varargin) 
        
        % Make CSV file ordering rows and columns by module membership
        makeCommunitiesCSV(obj,fileName,commLabels)
             
        %% Motif functions
        [tripletCounts tripletCountsRand] = tripletFrequency(obj,varargin)
        [doubletCounts doubletCountsRand] = doubletFrequency(obj,varargin)
        listTriplets(obj,motifNum,N,threshold)
        
        
        
        %% Functions to analyze degree
        %Functions to plot degree
        
        %Functions to fit degree
        [lambda p gof] = plFitDeg(obj, varargin)
        [lambda p gof] = expFitDeg(obj, varargin)
        h = plotDegree(obj, varargin)
       
        % Load distribution histogram
        h = loadDistribution(obj)
        
        %% Function for analysis of reciprocal connections
        [P Prand bins] = distrReciprocal(obj,binSize,nRand)
        calculateReciprocity(obj,N)
        P = recurrenceProb(obj,frac,varargin)
        
        %% Graph layout functions
        [A labels] = sequentialLayout(obj)
        
    end
    
    
end